<?php
/* --------------------------------------------------------------
   WithdrawalWriter.php 2019-11-08
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Withdrawal\Repository;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\ConnectionException;
use Exception;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalId;
use Gambio\Admin\Withdrawal\Interfaces\WithdrawalIds;
use Gambio\Admin\Withdrawal\Interfaces\Withdrawal;
use Gambio\Admin\Withdrawal\Interfaces\Withdrawals;

/**
 * Class WithdrawalWriter
 *
 * @package Gambio\Admin\Withdrawal\Repository
 */
class WithdrawalWriter
{
    /**
     * @var Connection
     */
    private $db;
    
    
    /**
     * ConfigurationQbReader constructor.
     *
     * @param Connection $db
     */
    public function __construct(Connection $db)
    {
        $this->db = $db;
    }
    
    
    /**
     * Inserts a new withdrawal into the database and returns the ID.
     *
     * @param Withdrawal $withdrawal
     *
     * @return int
     */
    public function insert(Withdrawal $withdrawal): int
    {
        $this->db->createQueryBuilder()
            ->insert('withdrawals')
            ->setValue('order_id', ':order_id')
            ->setValue('order_date', ':order_date')
            ->setValue('delivery_date', ':delivery_date')
            ->setValue('customer_id', ':customer_id')
            ->setValue('customer_gender', ':customer_gender')
            ->setValue('customer_firstname', ':customer_firstname')
            ->setValue('customer_lastname', ':customer_lastname')
            ->setValue('customer_street_address', ':customer_street_address')
            ->setValue('customer_postcode', ':customer_postcode')
            ->setValue('customer_city', ':customer_city')
            ->setValue('customer_country', ':customer_country')
            ->setValue('customer_email', ':customer_email')
            ->setValue('withdrawal_date', ':withdrawal_date')
            ->setValue('withdrawal_content', ':withdrawal_content')
            ->setValue('created_by_admin', ':created_by_admin')
            ->setValue('date_created', 'now()')
            ->setParameter('order_id', $withdrawal->orderId())
            ->setParameter('order_date', $withdrawal->orderCreationDate())
            ->setParameter('delivery_date', $withdrawal->orderDeliveryDate())
            ->setParameter('customer_id', $withdrawal->customerId())
            ->setParameter('customer_gender', $withdrawal->customerGender())
            ->setParameter('customer_firstname', $withdrawal->customerFirstName())
            ->setParameter('customer_lastname', $withdrawal->customerLastName())
            ->setParameter('customer_street_address', $withdrawal->customerStreet())
            ->setParameter('customer_postcode', $withdrawal->customerPostcode())
            ->setParameter('customer_city', $withdrawal->customerCity())
            ->setParameter('customer_country', $withdrawal->customerCountry())
            ->setParameter('customer_email', $withdrawal->customerEmail())
            ->setParameter('withdrawal_date', $withdrawal->date())
            ->setParameter('withdrawal_content', $withdrawal->content())
            ->setParameter('created_by_admin', $withdrawal->wasCreatedByAdmin() ? 1 : 0)
            ->execute();
        
        return (int)$this->db->lastInsertId();
    }
    
    
    /**
     * Inserts multiple withdrawals into the database and returns the ID.
     *
     * @param Withdrawals $withdrawals
     *
     * @return int[]
     *
     * @throws ConnectionException
     */
    public function insertMultiple(Withdrawals $withdrawals): array
    {
        $ids = [];
        try {
            $this->db->beginTransaction();
            foreach ($withdrawals as $reference => $withdrawal) {
                $ids[$reference] = $this->insert($withdrawal);
            }
            $this->db->commit();
        } catch (Exception $exception) {
            $this->db->rollBack();
            throw new \RuntimeException('Could not add withdrawal with reference "' . $reference . '".',
                                        $exception->getCode(),
                                        $exception);
        }
        
        return $ids;
    }
    
    
    /**
     * Updates an existing withdrawal in the database.
     *
     * @param Withdrawal $withdrawal
     */
    public function update(Withdrawal $withdrawal): void
    {
        $this->db->createQueryBuilder()
            ->update('withdrawals')
            ->set('order_id', ':order_id')
            ->set('order_date', ':order_date')
            ->set('delivery_date', ':delivery_date')
            ->set('customer_id', ':customer_id')
            ->set('customer_gender', ':customer_gender')
            ->set('customer_firstname', ':customer_firstname')
            ->set('customer_lastname', ':customer_lastname')
            ->set('customer_street_address', ':customer_street_address')
            ->set('customer_postcode', ':customer_postcode')
            ->set('customer_city', ':customer_city')
            ->set('customer_country', ':customer_country')
            ->set('customer_email', ':customer_email')
            ->set('withdrawal_date', ':withdrawal_date')
            ->set('withdrawal_content', ':withdrawal_content')
            ->set('created_by_admin', ':created_by_admin')
            ->where('withdrawal_id = :id')
            ->setParameter('order_id', $withdrawal->orderId())
            ->setParameter('order_date', $withdrawal->orderCreationDate())
            ->setParameter('delivery_date', $withdrawal->orderDeliveryDate())
            ->setParameter('customer_id', $withdrawal->customerId())
            ->setParameter('customer_gender', $withdrawal->customerGender())
            ->setParameter('customer_firstname', $withdrawal->customerFirstName())
            ->setParameter('customer_lastname', $withdrawal->customerLastName())
            ->setParameter('customer_street_address', $withdrawal->customerStreet())
            ->setParameter('customer_postcode', $withdrawal->customerPostcode())
            ->setParameter('customer_city', $withdrawal->customerCity())
            ->setParameter('customer_country', $withdrawal->customerCountry())
            ->setParameter('customer_email', $withdrawal->customerEmail())
            ->setParameter('withdrawal_date', $withdrawal->date())
            ->setParameter('withdrawal_content', $withdrawal->content())
            ->setParameter('created_by_admin', $withdrawal->wasCreatedByAdmin() ? 1 : 0)
            ->setParameter('id', $withdrawal->id())
            ->execute();
    }
    
    
    /**
     * Updates multiple existing withdrawals in the database.
     *
     * @param Withdrawals $withdrawals
     *
     * @throws ConnectionException
     */
    public function updateMultiple(Withdrawals $withdrawals): void
    {
        try {
            $this->db->beginTransaction();
            foreach ($withdrawals as $reference => $withdrawal) {
                $this->update($withdrawal);
            }
            $this->db->commit();
        } catch (Exception $exception) {
            $this->db->rollBack();
            throw new \RuntimeException('Could not update withdrawal with reference "' . $reference . '".',
                                        $exception->getCode(),
                                        $exception);
        }
    }
    
    
    /**
     * Deletes a withdrawal from the database.
     *
     * @param WithdrawalId $id
     */
    public function delete(WithdrawalId $id): void
    {
        $this->db->createQueryBuilder()
            ->delete('withdrawals')
            ->where('withdrawal_id = :id')
            ->setParameter('id', $id->value())
            ->execute();
    }
    
    
    /**
     * Deletes multiple withdrawals from the database.
     *
     * @param WithdrawalIds $ids
     *
     * @throws ConnectionException
     */
    public function deleteMultiple(WithdrawalIds $ids): void
    {
        try {
            $this->db->beginTransaction();
            foreach ($ids as $reference => $id) {
                $this->delete($id);
            }
            $this->db->commit();
        } catch (Exception $exception) {
            $this->db->rollBack();
            throw new \RuntimeException('Could not delete withdrawal with reference "' . $reference . '".',
                                        $exception->getCode(),
                                        $exception);
        }
    }
}